# ProofIn - Code Structure Documentation

## Overview

This plugin follows WordPress best practices with a clean, object-oriented architecture. The code is organized into separate classes with single responsibilities.

## Directory Structure

```
proofin/
├── proofin.php                    # Main plugin file (bootstrapper)
├── includes/                        # Core plugin classes
│   ├── class-plugin.php             # Main plugin controller
│   ├── class-database.php           # Database operations handler
│   ├── admin/                       # Admin-specific classes
│   │   ├── class-menu.php           # Admin menu registration
│   │   ├── class-assets.php         # Admin CSS/JS enqueuing
│   │   └── class-ajax-handler.php   # AJAX request handlers
│   └── frontend/                    # Frontend-specific classes
│       └── class-shortcode.php      # Shortcode handler
├── templates/                       # Template files
│   ├── admin/                       # Admin page templates
│   │   ├── sections-list.php        # Sections list page
│   │   └── add-new-section.php      # Add/edit section form
│   └── frontend/                    # Frontend templates
│       └── display-section.php      # Frontend rendering functions
└── assets/                          # Static assets
    ├── css/
    │   └── admin-style.css          # Admin styles
    ├── js/
    │   └── admin-script.js          # Admin JavaScript
    └── icons/                       # AI model icons
        ├── chatgpt.svg
        ├── claude.svg
        └── perplexity.svg
```

## Class Architecture

### 1. **ProofIn_Plugin** (`includes/class-plugin.php`)

- **Purpose**: Main controller class, singleton pattern
- **Responsibilities**:
  - Load all dependencies
  - Initialize components
  - Register WordPress hooks
  - Handle plugin activation
- **Key Methods**:
  - `get_instance()` - Get singleton instance
  - `load_dependencies()` - Load all required files
  - `init_components()` - Initialize all plugin components
  - `activate()` - Plugin activation handler

### 2. **ProofIn_Database** (`includes/class-database.php`)

- **Purpose**: Database operations handler
- **Responsibilities**:
  - Create and manage database tables
  - CRUD operations for sections
  - Data validation and sanitization
- **Key Methods**:
  - `create_table()` - Create plugin database table
  - `get_all_sections()` - Retrieve all sections
  - `get_section($id)` - Get single section
  - `insert_section($data)` - Insert new section
  - `update_section($id, $data)` - Update existing section
  - `delete_section($id)` - Delete section
  - `delete_sections($ids)` - Bulk delete sections

### 3. **ProofIn_Admin_Menu** (`includes/admin/class-menu.php`)

- **Purpose**: Admin menu management
- **Responsibilities**:
  - Register admin menu pages
  - Render admin page templates
- **Key Methods**:
  - `register_menu()` - Register WordPress admin menu
  - `render_sections_page()` - Render sections list page
  - `render_add_new_page()` - Render add/edit section page

### 4. **ProofIn_Admin_Assets** (`includes/admin/class-assets.php`)

- **Purpose**: Admin asset management
- **Responsibilities**:
  - Enqueue admin CSS and JavaScript
  - Localize scripts with data
- **Key Methods**:
  - `enqueue_assets($hook)` - Conditionally enqueue admin assets

### 5. **ProofIn_Ajax_Handler** (`includes/admin/class-ajax-handler.php`)

- **Purpose**: AJAX request handling
- **Responsibilities**:
  - Handle all AJAX requests
  - Validate nonces and permissions
  - Process and sanitize data
- **Key Methods**:
  - `save_section()` - Save (create/update) section
  - `delete_section()` - Delete section
  - `get_section()` - Retrieve section data

### 6. **ProofIn_Shortcode** (`includes/frontend/class-shortcode.php`)

- **Purpose**: Shortcode management
- **Responsibilities**:
  - Register `[proofin]` shortcode
  - Render frontend section output
- **Key Methods**:
  - `render($atts)` - Render shortcode output

## Data Flow

### Admin Section Save Flow

```
User Form Submit
    ↓
admin-script.js (AJAX)
    ↓
ProofIn_Ajax_Handler::save_section()
    ↓
ProofIn_Database::insert_section() or update_section()
    ↓
WordPress Database (wp_proofin_sections table)
```

### Frontend Shortcode Rendering Flow

```
WordPress Content Parser
    ↓
[proofin id="X"] Shortcode Detected
    ↓
ProofIn_Shortcode::render()
    ↓
ProofIn_Database::get_section()
    ↓
proofin_render_section_html() (in display-section.php)
    ↓
HTML Output to Page
```

## Best Practices Implemented

1. **Separation of Concerns**: Each class has a single, well-defined responsibility
2. **Dependency Injection**: Database instance passed to classes that need it
3. **Singleton Pattern**: Main plugin class uses singleton to prevent multiple instances
4. **Security**: All data sanitized and validated, nonce verification on AJAX requests
5. **Localization**: All strings wrapped in translation functions
6. **Hooks Organization**: Actions and filters registered in class constructors
7. **File Organization**: Clear separation between admin and frontend code
8. **Namespacing**: All classes prefixed with `ProofIn_` to avoid conflicts

## Database Schema

**Table**: `wp_proofin_sections`

| Column                 | Type         | Description                               |
| ---------------------- | ------------ | ----------------------------------------- |
| id                     | bigint(20)   | Primary key, auto-increment               |
| title                  | varchar(255) | Section title                             |
| template               | varchar(50)  | Template type (hero, card, split, footer) |
| section_heading        | varchar(255) | Heading text                              |
| section_description    | text         | Description text                          |
| prompt_text            | text         | AI prompt template                        |
| heading_font_size      | varchar(10)  | Heading font size (rem)                   |
| heading_font_weight    | varchar(10)  | Heading font weight                       |
| subheading_font_size   | varchar(10)  | Subheading font size (rem)                |
| subheading_font_weight | varchar(10)  | Subheading font weight                    |
| heading_color          | varchar(7)   | Heading color (hex)                       |
| subheading_color       | varchar(7)   | Subheading color (hex)                    |
| background_color       | varchar(7)   | Background color (hex)                    |
| ai_models              | text         | Comma-separated AI model list             |
| created_at             | datetime     | Creation timestamp                        |
| updated_at             | datetime     | Last update timestamp                     |

## Extending the Plugin

### Adding a New Admin Page

1. Add method in `ProofIn_Admin_Menu::register_menu()`
2. Create template file in `admin/` directory
3. Add render method in `ProofIn_Admin_Menu`

### Adding a New AJAX Handler

1. Add method in `ProofIn_Ajax_Handler`
2. Register action hook in constructor
3. Add JavaScript in `admin/js/admin-script.js`

### Adding a New Frontend Feature

1. Create new class in `includes/frontend/`
2. Initialize in `ProofIn_Plugin::init_components()`
3. Add frontend templates in `public/` directory

## Security Considerations

- ✅ Nonce verification on all AJAX requests
- ✅ Capability checks (`manage_options`)
- ✅ Data sanitization on input
- ✅ Data escaping on output
- ✅ Direct file access prevention
- ✅ SQL injection prevention (prepared statements)
- ✅ XSS prevention (esc\_\* functions)

## Performance Optimizations

- Only load admin assets on plugin pages
- Conditional component initialization (admin vs frontend)
- Database queries optimized with prepared statements
- Frontend CSS inlined to reduce HTTP requests
- Singleton pattern prevents duplicate initialization
