/**
 * WPProofIn Admin JavaScript - Tabbed Interface with Live Preview Version
 * @package WPProofIn
 */

jQuery(document).ready(function ($) {
  "use strict";

  // ==================== TAB SWITCHING ===================
  $(".proofin-tab-button").on("click", function () {
    const tabName = $(this).data("tab");

    // Update tab buttons
    $(".proofin-tab-button").removeClass("active");
    $(this).addClass("active");

    // Update tab panels
    $(".proofin-tab-panel").removeClass("active");
    $(`.proofin-tab-panel[data-panel="${tabName}"]`).addClass("active");
  });

  // ==================== DEVICE TOGGLE ====================
  $(".preview-device-btn").on("click", function () {
    const device = $(this).data("device");

    // Update buttons
    $(".preview-device-btn").removeClass("active");
    $(this).addClass("active");

    // Update preview frame
    $(".proofin-preview-frame").removeClass("desktop mobile").addClass(device);
  });

  // ==================== BACKGROUND TYPE TOGGLE ====================
  $("#background-type").on("change", function () {
    const type = $(this).val();
    if (type === "color") {
      $("#background-color-field").show();
      $("#background-image-field").hide();
      $("#background-image-preview-container").hide();
    } else {
      $("#background-color-field").hide();
      $("#background-image-field").show();
      // Show preview container if image exists
      if ($("#background-image").val()) {
        $("#background-image-preview-container").show();
      }
    }
    updateLivePreview();
  });

  // Initialize background type on page load
  $("#background-type").trigger("change");

  // ==================== MEDIA UPLOADER ====================
  $(".proofin-upload-image-btn").on("click", function (e) {
    e.preventDefault();

    // Get current image URL and ID if exists
    const currentImageUrl = $("#background-image").val();
    const currentImageId = $("#background-image-id").val();

    // Create a new media frame each time to ensure proper selection state
    const wpMediaFrame = wp.media({
      title: "Select Background Image",
      button: {
        text: "Use this image",
      },
      multiple: false,
    });

    // If there's a current image, pre-select it in the media library
    wpMediaFrame.on("open", function () {
      if (currentImageId) {
        const selection = wpMediaFrame.state().get("selection");
        const attachment = wp.media.attachment(currentImageId);

        // Wait for the attachment to be fetched before adding to selection
        attachment.fetch().done(function () {
          selection.add(attachment);
        });
      }
    });

    // When an image is selected, run a callback.
    wpMediaFrame.on("select", function () {
      const attachment = wpMediaFrame.state().get("selection").first().toJSON();

      // Store both URL and ID
      $("#background-image").val(attachment.url);
      $("#background-image-id").val(attachment.id);

      // Update preview
      const assetsUrl = proofin_admin.assets_url || "";
      $("#background-image-preview").html(
        '<img src="' +
          attachment.url +
          '" alt="Background">' +
          '<button type="button" class="proofin-remove-image">' +
          '<img src="' +
          proofin_admin.assets_url +
          'icons/cross.svg" alt="Remove">' +
          "</button>"
      );

      // Show preview container
      $("#background-image-preview-container").show();

      updateLivePreview();

      // Enable update button for existing sections
      const currentSectionId = $("#section-id").val();
      const isExistingSection =
        currentSectionId && currentSectionId !== "0" && currentSectionId !== "";

      if (isExistingSection && !formChanged) {
        formChanged = true;
        const $submitBtn = $("#proofin-section-form").find(
          'button[type="submit"]'
        );
        $submitBtn.prop("disabled", false).removeClass("disabled");
      }
    });

    // Finally, open the modal
    wpMediaFrame.open();
  });

  // Remove image
  $(document).on("click", ".proofin-remove-image", function (e) {
    e.preventDefault();
    $("#background-image").val("");
    $("#background-image-id").val("");
    $("#background-image-preview").html(
      '<div class="proofin-image-placeholder">' +
        '<span class="dashicons dashicons-format-image"></span>' +
        "<p>No image selected</p>" +
        "</div>"
    );
    $("#background-image-preview-container").hide();
    updateLivePreview();

    // Enable update button for existing sections
    const currentSectionId = $("#section-id").val();
    const isExistingSection =
      currentSectionId && currentSectionId !== "0" && currentSectionId !== "";

    if (isExistingSection && !formChanged) {
      formChanged = true;
      const $submitBtn = $("#proofin-section-form").find(
        'button[type="submit"]'
      );
      $submitBtn.prop("disabled", false).removeClass("disabled");
    }
  });

  // ==================== TEMPLATE SELECTION ====================
  $('input[name="template"]').on("change", function () {
    // Update visual checked state
    $(".proofin-template-option").removeClass("checked");
    $(this).closest(".proofin-template-option").addClass("checked");

    // Update live preview
    updateLivePreview();
  });

  // ==================== LIVE PREVIEW UPDATES ====================
  const previewFields = [
    "#section-heading",
    "#section-description",
    'input[name="template"]',
    "#heading-font-size",
    "#heading-font-weight",
    "#heading-color",
    "#subheading-font-size",
    "#subheading-font-weight",
    "#subheading-color",
    "#description-border-width",
    "#description-border-radius",
    "#description-border-color",
    "#background-type",
    "#background-color",
    "#background-image",
    'input[name="aiModels[]"]',
  ];

  // Bind change events for live preview
  previewFields.forEach(function (selector) {
    $(document).on("input change", selector, updateLivePreview);
  });

  // Initial preview render
  updateLivePreview();

  function updateLivePreview() {
    const template = $('input[name="template"]:checked').val() || "hero";
    const heading = $("#section-heading").val() || "Research Us Anywhere";
    const description =
      $("#section-description").val() ||
      "Learn more about our product using your favorite assistants and communities.";

    // Font styles
    const headingSize = $("#heading-font-size").val() || "2.5";
    const headingWeight = $("#heading-font-weight").val() || "700";
    const headingColor = $("#heading-color").val() || "#1e293b";
    const subheadingSize = $("#subheading-font-size").val() || "1.2";
    const subheadingWeight = $("#subheading-font-weight").val() || "400";
    const subheadingColor = $("#subheading-color").val() || "#64748b";
    const descriptionBorderWidth = parseInt(
      $("#description-border-width").val()
    );
    const descriptionBorderRadius = parseInt(
      $("#description-border-radius").val()
    );
    const descriptionBorderColor =
      $("#description-border-color").val() || "#000000";
    const backgroundType = $("#background-type").val() || "color";
    const backgroundColor = $("#background-color").val() || "#f8fafc";
    const backgroundImage = $("#background-image").val() || "";

    // AI Models
    const selectedModels = [];
    $('input[name="aiModels[]"]:checked').each(function () {
      selectedModels.push($(this).val());
    });

    // Use only selected models (no default)
    const models = selectedModels;

    // Generate preview HTML
    const previewHTML = generatePreviewHTML(template, {
      heading,
      description,
      headingSize,
      headingWeight,
      headingColor,
      subheadingSize,
      subheadingWeight,
      subheadingColor,
      descriptionBorderWidth,
      descriptionBorderRadius,
      descriptionBorderColor,
      backgroundType,
      backgroundColor,
      backgroundImage,
      models,
    });

    $("#proofin-live-preview").html(previewHTML);
  }

  function generatePreviewHTML(template, data) {
    const {
      heading,
      description,
      headingSize,
      headingWeight,
      headingColor,
      subheadingSize,
      subheadingWeight,
      subheadingColor,
      descriptionBorderWidth,
      descriptionBorderRadius,
      descriptionBorderColor,
      backgroundType,
      backgroundColor,
      backgroundImage,
      models,
    } = data;

    // Model colors
    const modelColors = {
      chatgpt: "#10a37f",
      claude: "#cc785c",
      perplexity: "#20808d",
      reddit: "#ff4500",
    };

    const modelNames = {
      chatgpt: "ChatGPT",
      claude: "Claude",
      perplexity: "Perplexity",
      reddit: "Reddit",
    };

    const modelIcons = {
      chatgpt: "chatgpt.svg",
      claude: "claude.svg",
      perplexity: "perplexity.svg",
      reddit: "reddit.svg",
    };

    // Get Assets URL from localized script
    const assetsUrl = proofin_admin.assets_url || "";

    // Generate AI model buttons matching frontend style (white background with borders)
    const modelButtons =
      models.length > 0
        ? models
            .map(
              (model) => `
            <a href="#" class="scs-ai-button" style="--model-color: ${modelColors[model]};" onclick="return false;">
                <img class="scs-ai-icon" src="${assetsUrl}icons/${modelIcons[model]}" alt="${modelNames[model]}" style="width: 24px; height: 24px;">
                <span class="scs-ai-name">${modelNames[model]}</span>
            </a>
        `
            )
            .join("")
        : "";

    // Background style
    let backgroundStyle = "";
    if (backgroundType === "image" && backgroundImage) {
      backgroundStyle = `background: linear-gradient(rgba(0,0,0,0.4), rgba(0,0,0,0.4)), url('${backgroundImage}') center/cover no-repeat;`;
    } else {
      backgroundStyle = `background-color: ${backgroundColor};`;
    }

    // Border style for section container
    let sectionBorderStyle = "";

    sectionBorderStyle = `border: ${descriptionBorderWidth}px solid ${descriptionBorderColor} !important; border-radius: ${descriptionBorderRadius}px !important;`;

    // Styles matching frontend CSS exactly
    const styles = `
            <style>
                /* Match frontend styling exactly */
                .scs-ai-section {
                    ${backgroundStyle}
                    ${sectionBorderStyle}
                    padding: 60px 20px;
                    text-align: center;
                    box-sizing: border-box;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
                }

                .scs-ai-container {
                    max-width: 1200px;
                    margin: 0 auto;
                }

                .scs-ai-heading {
                    font-size: ${headingSize}rem;
                    font-weight: ${headingWeight};
                    color: ${headingColor};
                    margin: 0 0 16px 0;
                    line-height: 1.2;
                }

                /* Override browser and WordPress defaults for h1 elements */
                h1.scs-ai-heading {
                    font-size: ${headingSize}rem !important;
                    font-weight: ${headingWeight} !important;
                }

                .scs-ai-description {
                    font-size: ${subheadingSize}rem;
                    font-weight: ${subheadingWeight};
                    color: ${subheadingColor};
                    line-height: 1.6;
                    margin: 0 auto 40px auto;
                    max-width: 600px;
                }

                .scs-ai-buttons {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 16px;
                    justify-content: center;
                    max-width: 800px;
                    margin: 0 auto;
                }   

                /* Button styles - white background with border like frontend */
                .scs-ai-button {
                    display: inline-flex !important;
                    align-items: center;
                    justify-content: center;
                    gap: 8px;
                    padding: 10px 16px;
                    background: #ffffff;
                    border: 2px solid #d1d5db;
                    border-radius: 16px;
                    text-decoration: none !important;
                    color: #1e293b;
                    font-weight: 600;
                    font-size: 1rem;
                    transition: all 0.32s cubic-bezier(0.2, 0.8, 0.2, 1);
                    position: relative;
                    overflow: hidden;
                    cursor: pointer;
                    min-width: 150px;
                }

                .scs-ai-button::before {
                    content: "";
                    position: absolute;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    background: var(--model-color);
                    opacity: 0;
                    transition: opacity 0.3s;
                }

                .scs-ai-button:hover {
                    border-color: var(--model-color);
                    transform: translateY(-6px);
                    box-shadow: 0 14px 28px rgba(0, 0, 0, 0.12);
                }

                .scs-ai-button:hover::before {
                    opacity: 0.06;
                }

                .scs-ai-icon {
                    position: relative;
                    z-index: 1;
                    width: 24px;
                    height: 24px;
                    flex-shrink: 0;
                }

                .scs-ai-name {
                    position: relative;
                    z-index: 1;
                    white-space: nowrap;
                }

                /* Hero Template */
                .scs-template-hero {
                    box-shadow: 0 6px 18px rgba(2, 6, 23, 0.06);
                    padding: 40px 32px;
                }

                .scs-template-hero .scs-ai-buttons {
                    gap: 12px;
                    margin-top: 24px;
                }

                .scs-template-hero .scs-ai-heading {
                    overflow-wrap: anywhere;
                }

                .scs-template-hero .scs-ai-description {
                    overflow-wrap: anywhere;
                }

                /* Card Template */
                .scs-template-card {
                    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
                    padding: 24px;
                }

                .scs-template-card .scs-ai-container {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    flex-wrap: wrap;
                    text-align: left;
                }

                .scs-template-card .scs-ai-heading {
                    /* Font size and weight are inherited from base .scs-ai-heading styles */
                    margin: 0 0 8px 0;
                    text-align: left;
                    overflow-wrap: anywhere;
                }

                .scs-template-card .scs-ai-description {
                    /* Font size and weight are inherited from base .scs-ai-description styles */
                    margin: 0;
                    text-align: left;
                    overflow-wrap: anywhere;
                }

                .scs-template-card .scs-ai-buttons {
                    margin: 20px 0 0 0;
                    flex: 1 1 100%;
                    justify-content: flex-start;
                }

                /* Footer Template */
                .scs-template-footer {
                    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
                    padding: 24px;
                }

                .scs-template-footer .scs-ai-heading {
                    overflow-wrap: anywhere;
                }

                .scs-template-footer .scs-ai-description {
                    overflow-wrap: anywhere;
                }

                .scs-template-footer .scs-ai-description {
                    /* Font size and weight are inherited from base .scs-ai-description styles */
                    margin: 0 auto 24px auto;
                }

                /* Icons Only Template */
                .scs-template-icons_only {
                    padding: 20px;
                }

                .scs-template-icons_only .scs-ai-heading,
                .scs-template-icons_only .scs-ai-description {
                    display: none;
                }

                .scs-template-icons_only .scs-ai-buttons {
                    gap: 16px;
                    max-width: none;
                }

                .scs-template-icons_only .scs-ai-button {
                    width: 48px !important;
                    height: 48px !important;
                    padding: 0 !important;
                    border-radius: 50%;
                    min-width: 48px !important;
                }

                .scs-template-icons_only .scs-ai-icon {
                    width: 28px !important;
                    height: 28px !important;
                }

                .scs-template-icons_only .scs-ai-name {
                    display: none;
                }

                /* Mobile Responsive - Match Frontend Behavior */
                @media screen and (max-width: 360px) {
                    .scs-ai-buttons {
                        flex-direction: column !important;
                        width: 100% !important;
                    }

                    .scs-ai-button {
                        width: 100% !important;
                        justify-content: center !important;
                    }
                }
            </style>
        `;

    // Template-specific layouts
    const buttonsHtml = modelButtons
      ? `<div class="scs-ai-buttons">${modelButtons}</div>`
      : "";

    switch (template) {
      case "card":
        return `
                    ${styles}
                    <div class="scs-ai-section scs-template-card">
                        <div class="scs-ai-container">
                            <div style="flex: 1 1 100%;">
                                <h3 class="scs-ai-heading">${heading}</h3>
                                <p class="scs-ai-description">${description}</p>
                            </div>
                            ${buttonsHtml}
                        </div>
                    </div>
                `;

      case "footer":
        return `
                    ${styles}
                    <div class="scs-ai-section scs-template-footer">
                        <div class="scs-ai-container">
                            <h3 class="scs-ai-heading">${heading}</h3>
                            <p class="scs-ai-description">${description}</p>
                            ${buttonsHtml}
                        </div>
                    </div>
                `;

      case "icons_only":
        return `
                    ${styles}
                    <div class="scs-ai-section scs-template-icons_only">
                        <div class="scs-ai-container">
                            ${buttonsHtml}
                        </div>
                    </div>
                `;

      case "hero":
      default:
        return `
                    ${styles}
                    <div class="scs-ai-section scs-template-hero">
                        <div class="scs-ai-container">
                            <h1 class="scs-ai-heading">${heading}</h1>
                            <p class="scs-ai-description">${description}</p>
                            ${buttonsHtml}
                        </div>
                    </div>
                `;
    }
  }

  // ==================== FORM CHANGE TRACKING ====================
  let formChanged = false;

  // Track changes in all form fields
  $("#proofin-section-form").on(
    "input change",
    "input, textarea, select",
    function () {
      const currentSectionId = $("#section-id").val();
      const isExistingSection =
        currentSectionId && currentSectionId !== "0" && currentSectionId !== "";

      if (isExistingSection && !formChanged) {
        formChanged = true;
        const $submitBtn = $("#proofin-section-form").find(
          'button[type="submit"]'
        );
        $submitBtn.prop("disabled", false).removeClass("disabled");
      }
    }
  );

  // ==================== FORM SUBMISSION ====================
  $("#proofin-section-form").on("submit", function (e) {
    e.preventDefault();
    console.log("Form submitted");

    const $submitBtn = $(this).find('button[type="submit"]');
    const originalText = $submitBtn.html();

    // Check if this is a new section (no ID) or an update
    const currentSectionId = $("#section-id").val();
    const isNewSection =
      !currentSectionId || currentSectionId === "0" || currentSectionId === "";

    // Disable submit button
    $submitBtn
      .prop("disabled", true)
      .html(
        '<span class="dashicons dashicons-update-alt" style="animation: rotation 2s infinite linear;"></span> Saving...'
      );

    // Collect form data
    const formData = {
      id: currentSectionId,
      title: $("#section-title").val(),
      template: $('input[name="template"]:checked').val(),
      sectionHeading: $("#section-heading").val(),
      sectionDescription: $("#section-description").val(),
      promptText: $("#prompt-text").val(),
      headingFontSize: $("#heading-font-size").val(),
      headingFontWeight: $("#heading-font-weight").val(),
      headingColor: $("#heading-color").val(),
      subheadingFontSize: $("#subheading-font-size").val(),
      subheadingFontWeight: $("#subheading-font-weight").val(),
      subheadingColor: $("#subheading-color").val(),
      descriptionBorderWidth: $("#description-border-width").val(),
      descriptionBorderRadius: $("#description-border-radius").val(),
      descriptionBorderColor: $("#description-border-color").val(),
      backgroundType: $("#background-type").val(),
      backgroundColor: $("#background-color").val(),
      backgroundImage: $("#background-image").val(),
      backgroundImageId: $("#background-image-id").val(),
      deviceDisplay: $('input[name="deviceDisplay"]:checked').val(),
      aiModels: $('input[name="aiModels[]"]:checked')
        .map(function () {
          return $(this).val();
        })
        .get(),
    };

    console.log("Form data:", formData);
    console.log("AJAX URL:", proofin_admin.ajax_url);

    // AJAX save
    $.ajax({
      url: proofin_admin.ajax_url,
      type: "POST",
      data: {
        action: "proofin_save_section",
        nonce: proofin_admin.nonce,
        section_data: formData,
      },
      success: function (response) {
        console.log("AJAX response:", response);
        if (response.success) {
          const sectionId = response.data.id;
          const shortcode = `[proofin id="${sectionId}"]`;

          if (isNewSection) {
            // For new sections: update section ID, change button text, and show popup
            $("#section-id").val(sectionId);

            // Update button text to "Update" and disable it (no changes made yet)
            $submitBtn
              .html('<span class="dashicons dashicons-yes"></span> Update')
              .prop("disabled", true)
              .addClass("disabled");

            // Reset form change tracking
            formChanged = false;

            // Set shortcode value
            $("#proofin-shortcode-input").val(shortcode);

            // Show modal
            $("#proofin-success-modal").fadeIn(300);
          } else {
            // For updates: just show success message, no popup
            showNotice("Section updated successfully!", "success");

            // Disable button again until changes are made
            $submitBtn
              .prop("disabled", true)
              .addClass("disabled")
              .html(originalText);

            // Reset form change tracking
            formChanged = false;
          }
        } else {
          console.error("Save failed:", response);
          showNotice(response.data.message || "Error saving section", "error");
          $submitBtn.prop("disabled", false).html(originalText);
        }
      },
      error: function (xhr, status, error) {
        console.error("AJAX error:", xhr, status, error);
        showNotice("Network error. Please try again.", "error");
        $submitBtn.prop("disabled", false).html(originalText);
      },
    });
  });

  // ==================== NOTICE DISPLAY ====================
  function showNotice(message, type) {
    const noticeClass = type === "success" ? "notice-success" : "notice-error";
    const notice = $(`
            <div class="notice ${noticeClass} is-dismissible" style="margin: 20px 0;">
                <p>${message}</p>
            </div>
        `);

    $(".proofin-admin-wrapper").prepend(notice);

    // Auto dismiss after 5 seconds
    setTimeout(function () {
      notice.fadeOut(function () {
        $(this).remove();
      });
    }, 5000);
  }

  // Add rotation animation for loading spinner
  $(
    "<style>@keyframes rotation { from { transform: rotate(0deg); } to { transform: rotate(359deg); } }</style>"
  ).appendTo("head");

  // ==================== SUCCESS MODAL HANDLERS ====================
  // Close modal when clicking X button
  $(".proofin-modal-close").on("click", function () {
    $("#proofin-success-modal").fadeOut(300);
  });

  // Close modal when clicking overlay
  $("#proofin-success-modal").on("click", function (e) {
    if ($(e.target).is("#proofin-success-modal")) {
      $(this).fadeOut(300);
    }
  });

  // Copy shortcode to clipboard
  $(".proofin-copy-button").on("click", function () {
    const $input = $("#proofin-shortcode-input");
    const $button = $(this);

    // Select and copy text
    $input[0].select();
    $input[0].setSelectionRange(0, 99999); // For mobile devices

    try {
      document.execCommand("copy");

      // Show success state
      $button.addClass("copied");
      const originalIcon = $button.html();
      $button.html('<span class="dashicons dashicons-yes"></span>');

      // Reset after 2 seconds
      setTimeout(function () {
        $button.removeClass("copied");
        $button.html(originalIcon);
      }, 2000);
    } catch (err) {
      console.error("Failed to copy:", err);
    }

    // Deselect
    window.getSelection().removeAllRanges();
  });
});
