<?php
/**
 * AJAX Handlers Class
 *
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class ProofIn_Ajax_Handler
 * Handles all AJAX requests
 */
class ProofIn_Ajax_Handler
{
    /**
     * Database instance
     *
     * @var ProofIn_Database
     */
    private $db;

    /**
     * Constructor
     *
     * @param ProofIn_Database $db Database instance
     */
    public function __construct($db)
    {
        $this->db = $db;

        add_action('wp_ajax_proofin_save_section', [$this, 'save_section']);
        add_action('wp_ajax_proofin_delete_section', [$this, 'delete_section']);
        add_action('wp_ajax_proofin_get_section', [$this, 'get_section']);
        add_action('wp_ajax_proofin_toggle_status', [$this, 'toggle_status']);
    }

    /**
     * Save section (create or update)
     */
    public function save_section()
    {
        // Verify nonce
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified on next line
        if (!isset($_POST['nonce']) || !wp_verify_nonce(wp_unslash($_POST['nonce']), 'proofin_nonce')) {
            wp_send_json_error(['message' => __('Invalid nonce', 'proofin')], 403);
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'proofin')], 403);
        }

        // Get section data (supports both nested and flat structures)
        $section_data = isset($_POST['section_data']) ? wp_unslash($_POST['section_data']) : wp_unslash($_POST);

        // Verify it's an array
        if (!is_array($section_data)) {
            wp_send_json_error(['message' => __('Invalid data format', 'proofin')], 400);
        }

        // Sanitize data
        $data = [
            'title' => sanitize_text_field($section_data['title'] ?? ''),
            'status' => isset($section_data['status']) ? absint($section_data['status']) : 1,
            'template' => sanitize_text_field($section_data['template'] ?? 'hero'),
            'section_heading' => sanitize_text_field($section_data['sectionHeading'] ?? ''),
            'section_description' => sanitize_textarea_field($section_data['sectionDescription'] ?? ''),
            'prompt_text' => sanitize_textarea_field($section_data['promptText'] ?? ''),
            'heading_font_size' => sanitize_text_field($section_data['headingFontSize'] ?? '2.5'),
            'heading_font_weight' => sanitize_text_field($section_data['headingFontWeight'] ?? '700'),
            'subheading_font_size' => sanitize_text_field($section_data['subheadingFontSize'] ?? '1.125'),
            'subheading_font_weight' => sanitize_text_field($section_data['subheadingFontWeight'] ?? '400'),
            'heading_color' => sanitize_hex_color($section_data['headingColor'] ?? '#1e293b'),
            'subheading_color' => sanitize_hex_color($section_data['subheadingColor'] ?? '#64748b'),
            'description_border_width' => sanitize_text_field($section_data['descriptionBorderWidth'] ?? '2'),
            'description_border_radius' => sanitize_text_field($section_data['descriptionBorderRadius'] ?? '20'),
            'description_border_color' => sanitize_hex_color($section_data['descriptionBorderColor'] ?? '#000000'),
            'background_color' => sanitize_hex_color($section_data['backgroundColor'] ?? '#f8fafc'),
            'background_type' => sanitize_text_field($section_data['backgroundType'] ?? 'color'),
            'background_image' => esc_url_raw($section_data['backgroundImage'] ?? ''),
            'background_image_id' => absint($section_data['backgroundImageId'] ?? 0),
            'ai_models' => isset($section_data['aiModels']) && is_array($section_data['aiModels'])
                ? implode(',', array_map('sanitize_text_field', $section_data['aiModels']))
                : '',
            'device_display' => sanitize_text_field($section_data['deviceDisplay'] ?? 'all')
        ];

        $section_id = isset($section_data['id']) ? absint($section_data['id']) : 0;

        if ($section_id > 0) {
            // Update existing section
            $result = $this->db->update_section($section_id, $data);
            $message = __('Section updated successfully!', 'proofin');
        } else {
            // Insert new section
            $result = $this->db->insert_section($data);
            $section_id = $result;
            $message = __('Section created successfully!', 'proofin');
        }

        if ($result !== false) {
            wp_send_json_success([
                'message' => $message,
                'id' => $section_id
            ]);
        } else {
            wp_send_json_error(['message' => __('Error saving section', 'proofin')], 500);
        }
    }

    /**
     * Delete section
     */
    public function delete_section()
    {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'proofin_nonce')) {
            wp_send_json_error(['message' => __('Invalid nonce', 'proofin')], 403);
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'proofin')], 403);
        }

        $section_id = isset($_POST['id']) ? absint($_POST['id']) : 0;

        if ($section_id === 0) {
            wp_send_json_error(['message' => __('Invalid section ID', 'proofin')], 400);
        }

        $result = $this->db->delete_section($section_id);

        if ($result) {
            wp_send_json_success(['message' => __('Section deleted successfully!', 'proofin')]);
        } else {
            wp_send_json_error(['message' => __('Error deleting section', 'proofin')], 500);
        }
    }

    /**
     * Get section data
     */
    public function get_section()
    {
        // Verify nonce
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified on next line
        if (!isset($_GET['nonce']) || !wp_verify_nonce(wp_unslash($_GET['nonce']), 'proofin_nonce')) {
            wp_send_json_error(['message' => __('Invalid nonce', 'proofin')], 403);
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'proofin')], 403);
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce already verified above
        $section_id = isset($_GET['id']) ? absint(wp_unslash($_GET['id'])) : 0;

        if ($section_id === 0) {
            wp_send_json_error(['message' => __('Invalid section ID', 'proofin')], 400);
        }

        $section = $this->db->get_section($section_id);

        if ($section) {
            wp_send_json_success($section);
        } else {
            wp_send_json_error(['message' => __('Section not found', 'proofin')], 404);
        }
    }

    /**
     * Toggle section status
     */
    public function toggle_status()
    {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'proofin_nonce')) {
            wp_send_json_error(['message' => __('Invalid nonce', 'proofin')], 403);
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'proofin')], 403);
        }

        $section_id = isset($_POST['section_id']) ? absint($_POST['section_id']) : 0;
        $status = isset($_POST['status']) ? absint($_POST['status']) : 0;

        if ($section_id === 0) {
            wp_send_json_error(['message' => __('Invalid section ID', 'proofin')], 400);
        }

        // Update status directly - simple update operation
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $this->db->update_section(
            $section_id,
            ['status' => $status]
        );

        if ($result !== false) {
            wp_send_json_success([
                'message' => __('Status updated successfully', 'proofin'),
                'status' => $status
            ]);
        } else {
            wp_send_json_error(['message' => __('Error updating status', 'proofin')], 500);
        }
    }
}
