<?php
/**
 * Admin Menu Handler Class
 *
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class ProofIn_Admin_Menu
 * Handles admin menu registration and page rendering
 */
class ProofIn_Admin_Menu
{
    /**
     * Constructor
     */
    public function __construct()
    {
        add_action('admin_menu', [$this, 'register_menu']);
        add_action('load-toplevel_page_proofin-sections', [$this, 'handle_delete_actions']);
    }

    /**
     * Handle delete actions before any output
     * This runs on load-{page} hook which executes before admin_enqueue_scripts
     */
    public function handle_delete_actions()
    {
        // Get database instance
        $db = ProofIn_Plugin::get_instance()->get_database();

        // Single delete
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified below via check_admin_referer
        if (isset($_GET['action']) && 'delete' === wp_unslash($_GET['action']) && isset($_GET['id'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified on next line
            $id = isset($_GET['id']) ? absint(wp_unslash($_GET['id'])) : 0;
            if ($id && check_admin_referer('delete_section_' . $id)) {
                $db->delete_section($id);
                wp_safe_redirect(admin_url('admin.php?page=proofin-sections&deleted=1'));
                exit;
            }
        }

        // Bulk delete
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified below via check_admin_referer
        if (isset($_POST['action']) && 'delete' === wp_unslash($_POST['action'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified on next line
            if (isset($_POST['section']) && is_array($_POST['section'])) {
                check_admin_referer('bulk-sections');
                $sections = array_map('absint', wp_unslash($_POST['section']));
                foreach ($sections as $id) {
                    $db->delete_section($id);
                }
                wp_safe_redirect(admin_url('admin.php?page=proofin-sections&deleted=' . count($sections)));
                exit;
            }
        }
    }

    /**
     * Register admin menu
     */
    public function register_menu()
    {
        // Main menu
        add_menu_page(
            __('ProofIn', 'proofin'),
            __('ProofIn', 'proofin'),
            'manage_options',
            'proofin-sections',
            [$this, 'render_sections_page'],
            'dashicons-format-chat',
            70
        );

        // All Sections submenu
        add_submenu_page(
            'proofin-sections',
            __('All Sections', 'proofin'),
            __('All Sections', 'proofin'),
            'manage_options',
            'proofin-sections',
            [$this, 'render_sections_page']
        );

        // Add New submenu
        add_submenu_page(
            'proofin-sections',
            __('Add New Section', 'proofin'),
            __('Add New', 'proofin'),
            'manage_options',
            'proofin-add-new',
            [$this, 'render_add_new_page']
        );
    }

    /**
     * Render sections list page from settings button
     */
    public function render_sections_page()
    {
        require_once PROOFIN_TEMPLATES_DIR . 'admin/sections-list.php';
    }

    /**
     * Render add/edit section page
     */
    public function render_add_new_page()
    {
        require_once PROOFIN_TEMPLATES_DIR . 'admin/add-new-section.php';
    }
}
