<?php
/**
 * Database Handler Class
 *
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class ProofIn_Database
 * Handles all database operations for sections
 */
class ProofIn_Database
{
    /**
     * Table name
     *
     * @var string
     */
    private $table_name;

    /**
     * Constructor
     */
    public function __construct()
    {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'proofin_sections';
    }

    /**
     * Create plugin database table
     */
    public function create_table()
    {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            title varchar(255) NOT NULL,
            status tinyint(1) DEFAULT 1,
            template varchar(50) NOT NULL DEFAULT 'hero',
            section_heading varchar(255) NOT NULL,
            section_description text,
            prompt_text text NOT NULL,
            heading_font_size varchar(10) DEFAULT '2.5',
            heading_font_weight varchar(10) DEFAULT '700',
            subheading_font_size varchar(10) DEFAULT '1.125',
            subheading_font_weight varchar(10) DEFAULT '400',
            heading_color varchar(7) DEFAULT '#1e293b',
            subheading_color varchar(7) DEFAULT '#64748b',
            description_border_width varchar(10) DEFAULT '2',
            description_border_radius varchar(10) DEFAULT '20',
            description_border_color varchar(7) DEFAULT '#000000',
            background_color varchar(7) DEFAULT '#f8fafc',
            background_type varchar(10) DEFAULT 'color',
            background_image text,
            background_image_id bigint(20) DEFAULT NULL,
            ai_models text,
            device_display varchar(20) DEFAULT 'all',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);

        // Run migrations to add missing columns for existing installations
        $this->migrate_add_background_image_id();
        $this->migrate_add_status_column();
    }

    /**
     * Migration: Add background_image_id column if it doesn't exist
     */
    private function migrate_add_background_image_id()
    {
        global $wpdb;

        // Check if column exists
        $column_exists = $wpdb->get_results(
            $wpdb->prepare(
                "SHOW COLUMNS FROM {$this->table_name} LIKE %s",
                'background_image_id'
            )
        );

        // Add column if it doesn't exist
        if (empty($column_exists)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->query(
                "ALTER TABLE {$this->table_name}
                ADD COLUMN background_image_id bigint(20) DEFAULT 0
                AFTER background_image"
            );
        }
    }

    /**
     * Migration: Add status column if it doesn't exist
     */
    private function migrate_add_status_column()
    {
        global $wpdb;

        // Check if column exists
        $column_exists = $wpdb->get_results(
            $wpdb->prepare(
                "SHOW COLUMNS FROM {$this->table_name} LIKE %s",
                'status'
            )
        );

        // Add column if it doesn't exist
        if (empty($column_exists)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->query(
                "ALTER TABLE {$this->table_name}
                ADD COLUMN status tinyint(1) DEFAULT 1
                AFTER title"
            );
        }
    }

    /**
     * Get all sections
     *
     * @param string $orderby Order by column
     * @param string $order ASC or DESC
     * @return array
     */
    public function get_all_sections($orderby = 'created_at', $order = 'DESC')
    {
        global $wpdb;

        // Whitelist allowed columns for ORDER BY
        $allowed_orderby = ['id', 'title', 'template', 'created_at', 'updated_at'];
        $orderby = in_array($orderby, $allowed_orderby, true) ? $orderby : 'created_at';

        // Whitelist allowed order direction
        $order = strtoupper($order) === 'ASC' ? 'ASC' : 'DESC';

        // Create cache key based on query parameters
        $cache_key = 'proofin_sections_all_' . $orderby . '_' . $order;
        $cache_group = 'proofin_sections';

        // Try to get cached results
        $results = wp_cache_get($cache_key, $cache_group);

        if (false === $results) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $orderby and $order are whitelisted above
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Caching implemented with wp_cache_get/set
            $results = $wpdb->get_results(
                "SELECT * FROM {$this->table_name} ORDER BY {$orderby} {$order}",
                ARRAY_A
            );

            // Cache the results
            wp_cache_set($cache_key, $results, $cache_group, 3600);
        }

        return $results;
    }

    /**
     * Get single section by ID
     *
     * @param int $id Section ID
     * @return array|null
     */
    public function get_section($id)
    {
        global $wpdb;

        // Create cache key for this section
        $cache_key = 'proofin_section_' . $id;
        $cache_group = 'proofin_sections';

        // Try to get cached result
        $result = wp_cache_get($cache_key, $cache_group);

        if (false === $result) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Caching implemented with wp_cache_get/set
            $result = $wpdb->get_row(
                $wpdb->prepare("SELECT * FROM {$this->table_name} WHERE id = %d", $id),
                ARRAY_A
            );

            // Cache the result
            if ($result) {
                wp_cache_set($cache_key, $result, $cache_group, 3600);
            }
        }

        return $result;
    }

    /**
     * Insert new section
     *
     * @param array $data Section data
     * @return int|false Insert ID or false on failure
     */
    public function insert_section($data)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Write operation, cache cleared after insert
        $result = $wpdb->insert(
            $this->table_name,
            $data,
            [
                '%s', // title
                '%d', // status
                '%s', // template
                '%s', // section_heading
                '%s', // section_description
                '%s', // prompt_text
                '%s', // heading_font_size
                '%s', // heading_font_weight
                '%s', // subheading_font_size
                '%s', // subheading_font_weight
                '%s', // heading_color
                '%s', // subheading_color
                '%s', // description_border_width
                '%s', // description_border_radius
                '%s', // description_border_color
                '%s', // background_color
                '%s', // background_type
                '%s', // background_image
                '%d', // background_image_id
                '%s', // ai_models
                '%s'  // device_display
            ]
        );

        if ($result) {
            // Clear all sections cache
            $this->clear_cache();
            return $wpdb->insert_id;
        }

        return false;
    }

    /**
     * Update existing section
     *
     * @param int $id Section ID
     * @param array $data Section data
     * @return bool
     */
    public function update_section($id, $data)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Write operation, cache cleared after update
        $result = $wpdb->update(
            $this->table_name,
            $data,
            ['id' => $id],
            [
                '%s', // title
                '%d', // status
                '%s', // template
                '%s', // section_heading
                '%s', // section_description
                '%s', // prompt_text
                '%s', // heading_font_size
                '%s', // heading_font_weight
                '%s', // subheading_font_size
                '%s', // subheading_font_weight
                '%s', // heading_color
                '%s', // subheading_color
                '%s', // description_border_width
                '%s', // description_border_radius
                '%s', // description_border_color
                '%s', // background_color
                '%s', // background_type
                '%s', // background_image
                '%d', // background_image_id
                '%s', // ai_models
                '%s'  // device_display
            ],
            ['%d'] // id
        );

        if (false !== $result) {
            // Clear cache for this section and all sections
            $this->clear_cache($id);
        }

        return $result;
    }

    /**
     * Delete section
     *
     * @param int $id Section ID
     * @return bool
     */
    public function delete_section($id)
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Write operation, cache cleared after delete
        $result = $wpdb->delete(
            $this->table_name,
            ['id' => $id],
            ['%d']
        );

        if ($result) {
            // Clear cache for this section and all sections
            $this->clear_cache($id);
        }

        return $result;
    }

    /**
     * Delete multiple sections
     *
     * @param array $ids Array of section IDs
     * @return int Number of deleted sections
     */
    public function delete_sections($ids)
    {
        global $wpdb;

        $count = 0;
        foreach ($ids as $id) {
            if ($this->delete_section((int) $id)) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Clear cache for sections
     *
     * @param int|null $id Optional section ID to clear specific cache
     */
    private function clear_cache($id = null)
    {
        $cache_group = 'proofin_sections';

        // Clear specific section cache if ID provided
        if ($id) {
            wp_cache_delete('proofin_section_' . $id, $cache_group);
        }

        // Clear all sections list cache (all possible sort combinations)
        $orderby_options = ['id', 'title', 'template', 'created_at', 'updated_at'];
        $order_options = ['ASC', 'DESC'];

        foreach ($orderby_options as $orderby) {
            foreach ($order_options as $order) {
                wp_cache_delete('proofin_sections_all_' . $orderby . '_' . $order, $cache_group);
            }
        }
    }
}
