<?php
/**
 * Main Plugin Class
 *
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class ProofIn_Plugin
 * Main plugin controller class
 */
class ProofIn_Plugin
{
    /**
     * Singleton instance
     *
     * @var ProofIn_Plugin
     */
    private static $instance = null;

    /**
     * Database handler
     *
     * @var ProofIn_Database
     */
    private $database;

    /**
     * Get singleton instance
     *
     * @return ProofIn_Plugin
     */
    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        $this->load_dependencies();
        $this->init_components();
        $this->register_hooks();
    }

    /**
     * Load required files
     */
    private function load_dependencies()
    {
        // Core classes
        require_once PROOFIN_INCLUDES_DIR . 'class-database.php';

        // Admin classes
        require_once PROOFIN_INCLUDES_DIR . 'admin/class-menu.php';
        require_once PROOFIN_INCLUDES_DIR . 'admin/class-assets.php';
        require_once PROOFIN_INCLUDES_DIR . 'admin/class-ajax-handler.php';

        // Frontend classes
        require_once PROOFIN_INCLUDES_DIR . 'frontend/class-render.php';
        require_once PROOFIN_INCLUDES_DIR . 'frontend/class-frontend-assets.php';
        require_once PROOFIN_INCLUDES_DIR . 'frontend/class-shortcode.php';
    }    /**
         * Initialize components
         */
    private function init_components()
    {
        // Initialize database
        $this->database = new ProofIn_Database();

        // Initialize admin components
        if (is_admin()) {
            new ProofIn_Admin_Menu();
            new ProofIn_Admin_Assets();
            new ProofIn_Ajax_Handler($this->database);
        }

        // Initialize frontend components
        if (!is_admin()) {
            new ProofIn_Frontend_Assets();
        }
        new ProofIn_Shortcode($this->database);
    }

    /**
     * Register WordPress hooks
     */
    private function register_hooks()
    {
        // Plugin action links - use dirname to get parent directory
        add_filter(
            'plugin_action_links_' . plugin_basename(dirname(__DIR__) . '/proofin.php'),
            [$this, 'add_action_links']
        );
    }

    /**
     * Plugin activation (static method called on activation)
     */
    public static function activate()
    {
        // Create database instance and table
        require_once PROOFIN_INCLUDES_DIR . 'class-database.php';
        $database = new ProofIn_Database();
        $database->create_table();
        flush_rewrite_rules();
    }

    /**
     * Add plugin action links
     *
     * @param array $links Existing links
     * @return array Modified links
     */
    public function add_action_links($links)
    {
        $settings_link = '<a href="' . admin_url('admin.php?page=proofin-sections') . '">' . __('Settings', 'proofin') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Get database instance
     *
     * @return ProofIn_Database
     */
    public function get_database()
    {
        return $this->database;
    }
}
