<?php
/**
 * Frontend Assets Handler
 *
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class ProofIn_Frontend_Assets
 * Handles frontend CSS enqueuing with proper detection
 */
class ProofIn_Frontend_Assets
{
    /**
     * Track if assets are enqueued
     *
     * @var bool
     */
    private static $enqueued = false;

    /**
     * Constructor
     */
    public function __construct()
    {
        add_action('wp_enqueue_scripts', [$this, 'register_styles']);
        add_action('wp_footer', [$this, 'enqueue_if_needed'], 5);
    }

    /**
     * Register styles early
     */
    public function register_styles()
    {
        wp_register_style('proofin-frontend', false, [], PROOFIN_VERSION);
    }

    /**
     * Enqueue styles only if shortcode is present
     */
    public function enqueue_if_needed()
    {
        if (self::$enqueued) {
            return;
        }

        global $post;

        // Check if shortcode exists in content
        if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'proofin')) {
            wp_enqueue_style('proofin-frontend');
            wp_add_inline_style('proofin-frontend', $this->get_css());
            self::$enqueued = true;
        }
    }

    /**
     * Get frontend CSS
     *
     * @return string CSS code
     */
    private function get_css()
    {
        return '
        /* ProofIn AI Section - Enhanced CSS Isolation */
        /* All selectors use #proofin-ai-section for higher specificity */
        /* This prevents site-wide CSS from affecting the plugin design */

        /* AI Prompts Section - Base Styles */
        #proofin-ai-section.scs-ai-section {
            padding: 60px 20px;
            text-align: center;
            box-sizing: border-box;
        }

        #proofin-ai-section * {
            box-sizing: border-box;
        }

        #proofin-ai-section .scs-ai-container {
            max-width: 1200px;
            margin: 0 auto;
        }

        #proofin-ai-section .scs-ai-heading {
            font-size: 2.5rem;
            font-weight: 700;
            margin: 0 0 16px 0;
            line-height: 1.2;
            overflow-wrap: anywhere;
        }

        #proofin-ai-section .scs-ai-description {
            font-size: 1.2rem;
            line-height: 1.6;
            margin: 0 auto 40px auto;
            max-width: 600px;
            overflow-wrap: anywhere;
        }

        #proofin-ai-section .scs-ai-buttons {
            display: flex;
            flex-wrap: wrap;
            gap: 16px;
            justify-content: center;
            max-width: 800px;
            margin: 0 auto;
        }

        /* Hero Template - Large centered */
        #proofin-ai-section.scs-template-hero {
            box-shadow: 0 6px 18px rgba(2, 6, 23, 0.06);
            padding: 40px 32px;
        }

        #proofin-ai-section.scs-template-hero .scs-ai-heading {
            font-size: 2.5rem;
            font-weight: 800;
        }

        #proofin-ai-section.scs-template-hero .scs-ai-buttons {
            gap: 12px;
            margin-top: 24px;
        }

        /* Card Template -  Left Aligned Content */
        #proofin-ai-section.scs-template-card {
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
            padding: 24px;
        }

        #proofin-ai-section.scs-template-card .scs-ai-container {
            display: flex;
            flex-direction: column;
            align-items: start;
            justify-content: space-between;
            flex-wrap: wrap;
            text-align: left;
        }

        #proofin-ai-section.scs-template-card .scs-ai-heading {
            font-size: 1.75rem;
            margin: 0 0 8px 0;
        }

        #proofin-ai-section.scs-template-card .scs-ai-description {
            margin: 0;
            text-align: left;
            font-size: 1rem;
        }

        #proofin-ai-section.scs-template-card .scs-ai-buttons {
            margin: 20px 0 0 0;
            flex: 1 1 100%;
            justify-content: flex-start;
        }

        /* Footer Template - Minimal compact */
        #proofin-ai-section.scs-template-footer {
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
            padding: 24px;
        }

        #proofin-ai-section.scs-template-footer .scs-ai-heading {
            font-size: 1.5rem;
            font-weight: 600;
        }

        #proofin-ai-section.scs-template-footer .scs-ai-description {
            font-size: 1rem;
            margin: 0 auto 24px auto;
        }

        /* Icons Only Template - Minimal horizontal */
        #proofin-ai-section.scs-template-icons_only {
            padding: 20px;
        }

        #proofin-ai-section.scs-template-icons_only .scs-ai-heading,
        #proofin-ai-section.scs-template-icons_only .scs-ai-description {
            display: none;
        }

        #proofin-ai-section.scs-template-icons_only .scs-ai-buttons {
            gap: 16px;
            max-width: none;
        }

        #proofin-ai-section.scs-template-icons_only .scs-ai-button {
            width: 48px !important;
            height: 48px !important;
            padding: 0 !important;
            border-radius: 50%;
            min-width: 48px !important;
        }

        #proofin-ai-section.scs-template-icons_only .scs-ai-icon {
            width: 28px !important;
            height: 28px !important;
        }

        #proofin-ai-section.scs-template-icons_only .scs-ai-name {
            display: none;
        }

        /* AI Button Styles - Enhanced with CSS Isolation */
        #proofin-ai-section .scs-ai-button {
            display: inline-flex !important;
            align-items: center;
            justify-content: center;
            gap: 8px;
            padding: 10px 16px;
            background: #ffffff;
            border: 2px solid #d1d5db;
            border-radius: 16px;
            text-decoration: none !important;
            color: #1e293b;
            font-weight: 600;
            font-size: 1rem;
            transition: all 0.32s cubic-bezier(0.2, 0.8, 0.2, 1);
            position: relative;
            overflow: hidden;
            box-sizing: border-box;
            vertical-align: middle;
        }

        #proofin-ai-section .scs-ai-button::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: var(--model-color);
            opacity: 0;
            transition: opacity 0.3s;
        }

        #proofin-ai-section .scs-ai-button:hover {
            border-color: var(--model-color);
            transform: translateY(-6px);
            box-shadow: 0 14px 28px rgba(0, 0, 0, 0.12);
        }

        #proofin-ai-section .scs-ai-button:hover::before {
            opacity: 0.06;
        }

        /* SVG Icon Styles - Critical isolation with !important */
        #proofin-ai-section .scs-ai-icon {
            position: relative;
            z-index: 1;
            width: 24px !important;
            height: 24px !important;
            min-width: 24px !important;
            min-height: 24px !important;
            max-width: 24px !important;
            max-height: 24px !important;
            display: inline-block !important;
            flex-shrink: 0 !important;
            object-fit: contain !important;
            vertical-align: middle !important;
            border: none !important;
            box-shadow: none !important;
            margin: 0 !important;
            padding: 0 !important;
        }

        #proofin-ai-section .scs-ai-name {
            position: relative;
            z-index: 1;
            white-space: nowrap;
        }

        /* Responsive Design - Enhanced Specificity */
        @media (max-width: 768px) {
            #proofin-ai-section.scs-ai-section {
                padding: 40px 16px;
            }

            #proofin-ai-section.scs-template-hero,
            #proofin-ai-section.scs-template-card,
            #proofin-ai-section.scs-template-split,
            #proofin-ai-section.scs-template-footer {
                padding: 32px 20px;
            }

            #proofin-ai-section .scs-ai-heading {
                font-size: 1.875rem;
            }

            #proofin-ai-section .scs-ai-description {
                font-size: 1rem;
            }

            #proofin-ai-section.scs-template-card .scs-ai-container {
                flex-direction: column;
                text-align: center;
            }

            #proofin-ai-section.scs-template-card .scs-ai-heading,
            #proofin-ai-section.scs-template-card .scs-ai-description {
                text-align: center;
            }

            #proofin-ai-section .scs-ai-button {
                padding: 12px 20px;
                font-size: 0.9375rem;
            }
        }

        @media (max-width: 360px) {
            #proofin-ai-section .scs-ai-buttons {
                flex-direction: column;
                width: 100%;
            }

            #proofin-ai-section .scs-ai-button {
                width: 100%;
            }

            #proofin-ai-section.scs-template-icons_only .scs-ai-buttons {
                flex-direction: row;
            }

            #proofin-ai-section.scs-template-icons_only .scs-ai-button {
                width: 48px !important;
            }
        }
        ';
    }
}
