<?php
/**
 * Frontend Rendering Class
 *
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class ProofIn_Render
 * Handles all frontend rendering logic
 */
class ProofIn_Render
{
    /**
     * Get AI model configurations
     *
     * @return array AI model configurations
     */
    public static function get_ai_models()
    {
        $models = [
            'chatgpt' => [
                'name' => 'ChatGPT',
                'url' => 'https://chat.openai.com/?q=',
                'icon' => 'chatgpt.svg',
                'color' => '#10a37f',
            ],
            'claude' => [
                'name' => 'Claude',
                'url' => 'https://claude.ai/new?q=',
                'icon' => 'claude.svg',
                'color' => '#cc785c',
            ],
            'perplexity' => [
                'name' => 'Perplexity',
                'url' => 'https://www.perplexity.ai/?q=',
                'icon' => 'perplexity.svg',
                'color' => '#20808d',
            ],
            'reddit' => [
                'name' => 'Reddit',
                'url' => 'https://www.reddit.com/search/?q=',
                'icon' => 'reddit.svg',
                'color' => '#ff4500',
            ],
        ];

        /**
         * Filter AI models
         *
         * @param array $models AI model configurations
         */
        return apply_filters('proofin_ai_models', $models);
    }

    /**
     * Render section HTML
     *
     * @param array $settings Section settings from database
     * @return string HTML output
     */
    public static function render_section_html($settings)
    {
        if (!is_array($settings)) {
            return '';
        }

        // Get site info for prompt replacement
        $site_name = get_bloginfo('name');
        $site_url = get_site_url();

        // Replace placeholders in prompt
        $prompt = $settings['promptText'];
        $prompt = str_replace('{site_name}', $site_name, $prompt);
        $prompt = str_replace('{site_url}', $site_url, $prompt);

        // URL encode the prompt
        $encoded_prompt = urlencode($prompt);

        // Get AI models
        $all_models = self::get_ai_models();
        $selected_models = !empty($settings['aiModels']) ? $settings['aiModels'] : [];

        // Build background style
        $background_type = $settings['backgroundType'] ?? 'color';
        $background_style = '';

        if ($background_type === 'image' && !empty($settings['backgroundImage'])) {
            $background_style = sprintf(
                'background: linear-gradient(rgba(0,0,0,0.4), rgba(0,0,0,0.4)), url(%s) center/cover no-repeat;',
                esc_url($settings['backgroundImage'])
            );
        } else {
            $background_style = sprintf(
                'background-color: %s;',
                esc_attr($settings['backgroundColor'])
            );
        }

        // Build border style for section container
        $border_width = intval($settings['descriptionBorderWidth']);
        $border_radius = intval($settings['descriptionBorderRadius']);
        $border_color = $settings['descriptionBorderColor'];

        $section_border_style = sprintf(
            'border: %dpx solid %s; border-radius: %dpx;',
            $border_width,
            esc_attr($border_color),
            $border_radius
        );

        $section_styles = $background_style . ' ' . $section_border_style;

        ob_start();

        /**
         * Action before section renders
         *
         * @param array $settings Section settings
         */
        do_action('proofin_before_section', $settings);
        ?>
        <div id="proofin-ai-section" class="scs-ai-section scs-template-<?php echo esc_attr($settings['template']); ?>"
            style="<?php echo esc_attr($section_styles); ?>">
            <div class="scs-ai-container">
                <?php if (!empty($settings['sectionHeading'])): ?>
                    <h3 class="scs-ai-heading"
                        style="font-size: <?php echo esc_attr($settings['headingFontSize']); ?>rem; font-weight: <?php echo esc_attr($settings['headingFontWeight']); ?>; color: <?php echo esc_attr($settings['headingColor']); ?>;">
                        <?php echo esc_html($settings['sectionHeading']); ?>
                    </h3>
                <?php endif; ?>

                <?php if (!empty($settings['sectionDescription'])): ?>
                    <p class="scs-ai-description"
                        style="font-size: <?php echo esc_attr($settings['subheadingFontSize']); ?>rem; font-weight: <?php echo esc_attr($settings['subheadingFontWeight']); ?>; color: <?php echo esc_attr($settings['subheadingColor']); ?>;">
                        <?php echo esc_html($settings['sectionDescription']); ?>
                    </p>
                <?php endif; ?>

                <?php if (!empty($selected_models)): ?>
                    <div class="scs-ai-buttons">
                        <?php foreach ($selected_models as $model_key): ?>
                            <?php if (isset($all_models[$model_key])):
                                $model = $all_models[$model_key];
                                $model_url = $model['url'] . $encoded_prompt;
                                ?>
                                <a href="<?php echo esc_url($model_url); ?>" class="scs-ai-button" target="_blank" rel="noopener noreferrer"
                                    style="--model-color: <?php echo esc_attr($model['color']); ?>">
                                    <img class="scs-ai-icon" src="<?php echo esc_url(PROOFIN_ASSETS_URL . 'icons/' . $model['icon']); ?>"
                                        alt="<?php echo esc_attr($model['name']); ?> <?php esc_attr_e('Icon', 'proofin'); ?>">
                                    <span class="scs-ai-name"><?php echo esc_html($model['name']); ?></span>
                                </a>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
        /**
         * Action after section renders
         *
         * @param array $settings Section settings
         */
        do_action('proofin_after_section', $settings);

        $output = ob_get_clean();

        /**
         * Filter section HTML output
         *
         * @param string $output HTML output
         * @param array $settings Section settings
         */
        return apply_filters('proofin_section_output', $output, $settings);
    }
}
