<?php
/**
 * Shortcode Handler Class
 *
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class ProofIn_Shortcode
 * Handles shortcode registration and rendering
 */
class ProofIn_Shortcode
{
    /**
     * Database instance
     *
     * @var ProofIn_Database
     */
    private $db;

    /**
     * Constructor
     *
     * @param ProofIn_Database $db Database instance
     */
    public function __construct($db)
    {
        $this->db = $db;

        add_shortcode('proofin', [$this, 'render']);
    }

    /**
     * Render shortcode
     *
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public function render($atts)
    {
        $atts = shortcode_atts([
            'id' => 0
        ], $atts);

        $section_id = absint($atts['id']);

        if ($section_id === 0) {
            return '<p>' . __('Invalid section ID', 'proofin') . '</p>';
        }

        $section = $this->db->get_section($section_id);

        if (!$section) {
            return '<p>' . __('Section not found', 'proofin') . '</p>';
        }

        // Check if section is enabled (status = 1)
        $status = isset($section['status']) ? absint($section['status']) : 1;
        if ($status !== 1) {
            return ''; // Don't display disabled sections
        }

        // Check device display settings
        $device_display = $section['device_display'] ?? 'all';

        if ($device_display !== 'all') {
            $is_mobile = wp_is_mobile();

            // If setting is "mobile" but current device is desktop, don't show
            if ($device_display === 'mobile' && !$is_mobile) {
                return '';
            }

            // If setting is "desktop" but current device is mobile, don't show
            if ($device_display === 'desktop' && $is_mobile) {
                return '';
            }
        }

        // Convert database data to settings format
        $settings = [
            'id' => $section_id, // Include section ID for unique element ID
            'template' => $section['template'],
            'sectionHeading' => $section['section_heading'],
            'sectionDescription' => $section['section_description'],
            'promptText' => $section['prompt_text'],
            'headingFontSize' => $section['heading_font_size'],
            'headingFontWeight' => $section['heading_font_weight'],
            'subheadingFontSize' => $section['subheading_font_size'],
            'subheadingFontWeight' => $section['subheading_font_weight'],
            'headingColor' => $section['heading_color'],
            'subheadingColor' => $section['subheading_color'],
            'descriptionBorderWidth' => $section['description_border_width'],
            'descriptionBorderRadius' => $section['description_border_radius'],
            'descriptionBorderColor' => $section['description_border_color'],
            'backgroundColor' => $section['background_color'],
            'backgroundType' => $section['background_type'] ?? 'color',
            'backgroundImage' => $section['background_image'] ?? '',
            'aiModels' => !empty($section['ai_models']) ? explode(',', $section['ai_models']) : []
        ];

        // Use the new render class
        return ProofIn_Render::render_section_html($settings);
    }
}
