<?php
/**
 * Sections List Page - Display all sections in a WordPress table
 * 
 * @package ProofIn
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Load WP_List_Table if not loaded
if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

/**
 * ProofIn_Sections_Table class
 */
class ProofIn_Sections_Table extends WP_List_Table
{
    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct([
            'singular' => __('Section', 'proofin'),
            'plural' => __('Sections', 'proofin'),
            'ajax' => false
        ]);
    }

    /**
     * Get columns
     */
    public function get_columns()
    {
        return [
            'cb' => '<input type="checkbox" />',
            'title' => __('Title', 'proofin'),
            'shortcode' => __('Shortcode', 'proofin'),
            'status' => __('Status', 'proofin'),
            'template' => __('Template', 'proofin'),
            'ai_models' => __('Platforms', 'proofin'),
            'devices' => __('Devices', 'proofin'),
            'created_at' => __('Created', 'proofin'),
        ];
    }

    /**
     * Get sortable columns
     */
    public function get_sortable_columns()
    {
        return [
            'title' => ['title', true],
            'template' => ['template', false],
            'created_at' => ['created_at', false],
        ];
    }

    /**
     * Checkbox column
     */
    public function column_cb($item)
    {
        return sprintf('<input type="checkbox" name="section[]" value="%s" />', esc_attr($item['id']));
    }

    /**
     * Title column with row actions
     */
    public function column_title($item)
    {
        $edit_url = admin_url('admin.php?page=proofin-add-new&id=' . $item['id']);
        $delete_url = wp_nonce_url(
            admin_url('admin.php?page=proofin-sections&action=delete&id=' . $item['id']),
            'delete_section_' . $item['id']
        );

        $actions = [
            'edit' => sprintf('<a href="%s">%s</a>', esc_url($edit_url), esc_html__('Edit', 'proofin')),
            'delete' => sprintf(
                '<a href="%s" class="submitdelete" onclick="return confirm(\'%s\');">%s</a>',
                esc_url($delete_url),
                esc_js(__('Are you sure you want to delete this section?', 'proofin')),
                esc_html__('Delete', 'proofin')
            ),
        ];

        return sprintf(
            '<strong><a href="%s">%s</a></strong>%s',
            esc_url($edit_url),
            esc_html($item['title']),
            $this->row_actions($actions)
        );
    }

    /**
     * Shortcode column
     */
    public function column_shortcode($item)
    {
        $shortcode = '[proofin id="' . $item['id'] . '"]';
        return sprintf(
            '<code class="proofin-shortcode-copy" onclick="proofinCopyShortcode(this)">%s</code>',
            esc_html($shortcode)
        );
    }

    /**
     * Template column
     */
    public function column_template($item)
    {
        $templates = [
            'hero' => __('Large Hero', 'proofin'),
            'card' => __('Soft Card', 'proofin'),
            'split' => __('Split Layout', 'proofin'),
            'footer' => __('Minimal Footer', 'proofin'),
            'icons_only' => __('Header/Footer Icons', 'proofin'),
        ];
        return isset($templates[$item['template']]) ? esc_html($templates[$item['template']]) : esc_html($item['template']);
    }

    /**
     * Status column with toggle switch
     */
    public function column_status($item)
    {
        $checked = !empty($item['status']) ? 'checked' : '';
        $status_class = !empty($item['status']) ? 'status-on' : 'status-off';

        return sprintf(
            '<label class="proofin-status-toggle">
                <input type="checkbox"
                       class="proofin-status-checkbox"
                       data-section-id="%d"
                       %s>
                <span class="proofin-status-slider %s"></span>
            </label>',
            absint($item['id']),
            $checked,
            esc_attr($status_class)
        );
    }

    /**
     * AI Models column
     */
    public function column_ai_models($item)
    {
        if (empty($item['ai_models'])) {
            return '—';
        }
        $models = explode(',', $item['ai_models']);
        $formatted = array_map(function ($model) {
            $names = [
                'chatgpt' => 'ChatGPT',
                'claude' => 'Claude',
                'perplexity' => 'Perplexity'
            ];
            return isset($names[$model]) ? $names[$model] : ucfirst($model);
        }, $models);
        return esc_html(implode(', ', $formatted));
    }

    /**
     * Devices column
     */
    public function column_devices($item)
    {
        $device_display = $item['device_display'] ?? 'all';

        $devices = [
            'all' => __('Show on All Devices', 'proofin'),
            'desktop' => __('Desktop Only', 'proofin'),
            'mobile' => __('Mobile Only', 'proofin')
        ];

        return isset($devices[$device_display]) ? esc_html($devices[$device_display]) : esc_html(ucfirst($device_display));
    }

    /**
     * Created date column
     */
    public function column_created_at($item)
    {
        return date_i18n(get_option('date_format'), strtotime($item['created_at']));
    }

    /**
     * Default column
     */
    public function column_default($item, $column_name)
    {
        return isset($item[$column_name]) ? esc_html($item[$column_name]) : '—';
    }

    /**
     * Get bulk actions
     */
    public function get_bulk_actions()
    {
        return [
            'delete' => __('Delete', 'proofin')
        ];
    }

    /**
     * Process bulk actions
     * Note: Actual deletion is handled in admin_init hook to prevent "headers already sent" errors
     */
    public function process_bulk_action()
    {
        // Delete actions are now handled in ProofIn_Admin_Menu::handle_delete_actions()
        // which runs on admin_init before any output is sent
    }

    /**
     * Prepare items
     */
    public function prepare_items()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'proofin_sections';

        $per_page = 20;
        $current_page = $this->get_pagenum();
        $offset = ($current_page - 1) * $per_page;

        // Get sorting parameters
        $orderby = isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'created_at';
        $order = isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'DESC';

        // Validate orderby
        $allowed_orderby = ['title', 'template', 'created_at'];
        if (!in_array($orderby, $allowed_orderby)) {
            $orderby = 'created_at';
        }

        // Validate order
        $order = strtoupper($order) === 'ASC' ? 'ASC' : 'DESC';

        // Get total items
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        // Get items - ORDER BY cannot be prepared, but we've validated it above
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $orderby and $order are validated against whitelist
        $this->items = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name ORDER BY $orderby $order LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        // Set pagination
        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page' => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ]);

        $columns = $this->get_columns();
        $hidden = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [$columns, $hidden, $sortable];

        $this->process_bulk_action();
    }
}

// Create table instance
$proofin_sections_table = new ProofIn_Sections_Table();
$proofin_sections_table->prepare_items();
?>

<div class="wrap">
    <h1 class="wp-heading-inline">
        <?php echo esc_html(get_admin_page_title()); ?>
    </h1>
    <a href="<?php echo esc_url(admin_url('admin.php?page=proofin-add-new')); ?>" class="page-title-action">
        <?php esc_html_e('Add New', 'proofin'); ?>
    </a>

    <?php if (isset($_GET['deleted'])): ?>
        <div class="notice notice-success is-dismissible">
            <p>
                <?php
                $proofin_count = absint($_GET['deleted']);
                printf(
                    esc_html('%d section deleted.', '%d sections deleted.', $proofin_count, 'proofin'),
                    esc_html($proofin_count)
                );
                ?>
            </p>
        </div>
    <?php endif; ?>

    <hr class="wp-header-end">

    <form method="post">
        <?php $proofin_sections_table->display(); ?>
    </form>

    <style>
        .proofin-shortcode-copy {
            cursor: pointer;
            padding: 4px 8px;
            background: #f0f0f1;
            border-radius: 3px;
            user-select: all;
            font-size: 13px;
        }

        .proofin-shortcode-copy:hover {
            background: #dcdcde;
        }

        .proofin-shortcode-copy:active {
            background: #c3c4c7;
        }

        /* Status Toggle Switch */
        .proofin-status-toggle {
            position: relative;
            display: inline-block;
            width: 44px;
            height: 20px;
            margin: 0;
        }

        .proofin-status-toggle input {
            opacity: 0;
            width: 0;
            height: 0;
        }

        .proofin-status-slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #b1b1b1;
            transition: 0.3s;
            border-radius: 24px;
        }

        .proofin-status-slider:before {
            position: absolute;
            content: "";
            height: 13px;
            width: 13px;
            left: 6px;
            bottom: 4px;
            background-color: white;
            transition: 0.3s;
            border-radius: 50%;
        }

        .proofin-status-checkbox:checked+.proofin-status-slider {
            background-color: #2271b1;
        }

        .proofin-status-checkbox:checked+.proofin-status-slider:before {
            transform: translateX(20px);
        }

        .proofin-status-slider:hover {
            opacity: 0.9;
        }
    </style>

    <script>
        function proofinCopyShortcode(element) {
            // Select the text
            const range = document.createRange();
            range.selectNode(element);
            window.getSelection().removeAllRanges();
            window.getSelection().addRange(range);

            // Copy to clipboard
            try {
                document.execCommand('copy');
                const originalText = element.textContent;
                element.textContent = '<?php esc_html_e('Copied!', 'proofin'); ?>';
                element.style.background = '#00a32a';
                element.style.color = '#fff';

                setTimeout(function () {
                    element.textContent = originalText;
                    element.style.background = '';
                    element.style.color = '';
                }, 1500);
            } catch (err) {
                console.error('Failed to copy shortcode:', err);
            }

            window.getSelection().removeAllRanges();
        }

        // Handle status toggle
        jQuery(document).ready(function ($) {
            $('.proofin-status-checkbox').on('change', function () {
                const checkbox = $(this);
                const sectionId = checkbox.data('section-id');
                const newStatus = checkbox.is(':checked') ? 1 : 0;

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'proofin_toggle_status',
                        nonce: '<?php echo esc_js(wp_create_nonce('proofin_nonce')); ?>',
                        section_id: sectionId,
                        status: newStatus
                    },
                    success: function (response) {
                        if (response.success) {
                            // Update slider class
                            const slider = checkbox.next('.proofin-status-slider');
                            if (newStatus) {
                                slider.removeClass('status-off').addClass('status-on');
                            } else {
                                slider.removeClass('status-on').addClass('status-off');
                            }
                        } else {
                            // Revert checkbox if failed
                            checkbox.prop('checked', !newStatus);
                            alert('<?php esc_html_e('Failed to update status', 'proofin'); ?>');
                        }
                    },
                    error: function () {
                        // Revert checkbox on error
                        checkbox.prop('checked', !newStatus);
                        alert('<?php esc_html_e('Error updating status', 'proofin'); ?>');
                    }
                });
            });
        });
    </script>
</div>